using System;
using System.Collections;
using System.Diagnostics;

using Bertaccini.Utils;

namespace Team_Project.PersistencyManagers.Protocols
{
	/// <summary>
	/// Gestore dei mutex per il controllo della concorrenza sui file
	/// </summary>
	public class MutexHolder
	{
		/// <summary>
		/// Oggetto di sincronizzazione. Ognuno di questi oggetti mantiene informazioni
		/// riguardanti un elmento che  acceduto.
		/// </summary>
		protected class SyncObj
		{
			/// <summary>
			/// Costruisce un SyncObj con il nome specificato
			/// </summary>
			/// <param name="name">Nome dell'oggetto</param>
			public SyncObj(string name)
			{
				WriterM = new Mutex2(1,name);
				RdrCount = 0;
			}

			/// <summary>
			/// Mutex di scrittura.
			/// </summary>
			public Mutex2 WriterM;
			/// <summary>
			/// Numero di processi che stanno leggendo il file
			/// </summary>
			public int RdrCount;
		}

		/// <summary>
		/// Hashtable contenente le collezioni di oggetti di sincronizzazione.
		/// La chiave  il nome del progetto
		/// </summary>
		protected Hashtable projects;
		/// <summary>
		/// Costruisce un mutexHolder vuoto
		/// </summary>
		public MutexHolder()
		{
			projects = new Hashtable();
		}

		/// <summary>
		/// Costruisce la chiave utilizzata dalle hash table per un certo elemento
		/// </summary>
		/// <param name="location">Location dell'elemento</param>
		/// <param name="name">Nome dell'elemetno</param>
		/// <returns>stringa da utilizzare come chiave della hashtable</returns>
		protected string BuildKey(string location,string name)
		{
			return location+" # "+name;// + " # " + dir.ToString();
		}

		/// <summary>
		/// Ottiene l'oggetto su cui sincronizzarsi per un certo elemento
		/// </summary>
		/// <param name="project">Nome del progetto</param>
		/// <param name="location">Location dell'elemento</param>
		/// <param name="name">Nome dell'elemento</param>
		/// <returns>Oggetto sulla quale effettuare la sincronizzazione</returns>
		protected SyncObj Get(string project,string location,string name)
		{
			lock(this)
			{
				if(!projects.ContainsKey(project))
				{
					Trace.WriteLine("MutexHolder: Creating Mutex table for: " + project);
					projects.Add(project,new Hashtable());
				}
				Hashtable projT = (Hashtable)projects[project];
				string Key = BuildKey(location,name);
				if(!projT.ContainsKey(Key))
				{
					projT.Add(Key,new SyncObj(Key));
				}
				return (SyncObj)projT[Key];
			}
		}
	
		/// <summary>
		/// Funzione da richiamare quando si desidera accedere in modo thread safe
		/// ad un elemento
		/// </summary>
		/// <param name="project">Nome del Progetto</param>
		/// <param name="location">Location dell'elemento</param>
		/// <param name="name">Nome dell'elemento</param>
		public void RequestRead(string project,string location,string name)
		{
			Trace.WriteLine("ReqestRead-"+project+" "+location+" "+name);
			SyncObj sync = Get(project,location,name);
			do
			{
				lock(sync)
				{
					if(sync.WriterM.Avaiable || sync.RdrCount > 0)
					{
						sync.RdrCount++;
						Trace.WriteLine("No writers, reader #" + sync.RdrCount.ToString());
						if(sync.RdrCount == 1)
						{
							Trace.WriteLine("First reader: blocking writers");
							sync.WriterM.Wait();
						}
						return;
					}
				}
				Trace.WriteLine("Writer detected, suspending");
				sync.WriterM.Wait();
				Trace.WriteLine("Awaking, retrying to get write mutex");
			}while (true);
		}

		/// <summary>
		/// Funzione da richiamare quando si  terminata l'operazione di lettura
		/// </summary>
		/// <param name="project">Nome del Progetto</param>
		/// <param name="location">Location dell'elemento</param>
		/// <param name="name">Nome dell'elemento</param>
		public void ReleaseRead(string project,string location,string name)
		{
			Trace.WriteLine("ReleaseRead-"+project+" "+location+" "+name);
			SyncObj sync = Get(project,location,name);
			lock(sync)
			{
				Trace.WriteLine("Reader #" + sync.RdrCount.ToString() + " finihed reading");
				sync.RdrCount--;
				if(sync.RdrCount == 0)
				{
					Trace.WriteLine("Last reader: releasing writers");
					sync.WriterM.Signal();
				}
			}
		}

		/// <summary>
		/// Riconosce se ci sono processi in attesa di scrittura.
		/// </summary>
		/// <param name="project">Nome del Progetto</param>
		/// <param name="location">Location dell'elemento</param>
		/// <param name="name">Nome dell'elemento</param>
		/// <returns>True se ci sono processi in attesa di scrittura,
		/// false altrimenti</returns>
		public bool AreWritersWaiting(string project,string location,string name)
		{
			SyncObj sync = Get(project,location,name);
			return sync.WriterM.WaitingCount > 0;
		}

		/// <summary>
		/// Richiede l'accesso in scrittura ad un elemento
		/// </summary>
		/// <param name="project">Nome del Progetto</param>
		/// <param name="location">Location dell'elemento</param>
		/// <param name="name">Nome dell'elemento</param>
		public void RequestWrite(string project,string location,string name)
		{
			Trace.WriteLine("RequestWrite-"+project+" "+location+" "+name);
			SyncObj sync = Get(project,location,name);
			sync.WriterM.Wait();
			Trace.WriteLine("WriteMutex aquired by writer.");
		}

		/// <summary>
		/// Rilascia la risorsa che era stata acquisita in scrittura
		/// </summary>
		/// <param name="project">Nome del Progetto</param>
		/// <param name="location">Location dell'elemento</param>
		/// <param name="name">Nome dell'elemento</param>
		public void ReleaseWrite(string project,string location,string name)
		{
			Trace.WriteLine("ReleaseWrite-"+project+" "+location+" "+name);
			SyncObj sync = Get(project,location,name);
			sync.WriterM.Signal();
			Trace.WriteLine("Writer released WriteMutex");			
		}
	}
}
